#!/bin/bash
# Copyright 2018 Badu Networks, Inc.
# FTP Test Script
revision="1.2"

# Exit codes
#  0 = Success
#  1 = Issue with provided parameters

function usage() {
   echo "Badu Networks FTP Test Script, version $revision"
   echo "Usage:  $0 --fileSize 10k --warpHost 192.168.0.1 --warpPort 8000 --standardHost 192.168.0.1 --standardPort 8001 --ftpUser myUser --ftpPassword myPassword [--direction down] [--count 20] [--delay 0.2]"
   echo -e "\nThis script will perform FTP transfers using cURL against the provided hosts, alternating between the two. Output is given in CSV format. Results for specified warpHost will be labeled WarpTCP, and results for standardHost will be labeled TCP."
   echo -e "\t--fileSize : Specify the file size to transfer.  On Badu test servers, the following file sizes are typically available: 5k, 10k, 50k, 100k, 500k, 1m, 5m, 10m"
   echo -e "\t--warpHost : IP address or hostname of the optimized path"
   echo -e "\t--warpPort : Destination TCP port for the optimized path"
   echo -e "\t--standardHost : IP address or hostname of the bypass path"
   echo -e "\t--standardPort : Destination TCP port for the bypass path"
   echo -e "\t--ftpUser : The FTP username for your test server"
   echo -e "\t--ftpPassword : The FTP password for your test server"
   echo -e "\t--direction [OPTIONAL] : Specify the file transfer direction.  Options are either up or down.  Script defaults to down if this flag is not included."
   echo -e "\t--count [OPTIONAL] : Number of iterations. Defaults to 10 if not specified."
   echo -e "\t--delay [OPTIONAL]: Delay between file transfers, in seconds.  Defaults to 0.1 seconds if not specified."
   echo -e "\t--help : Display this help information"
}


# Set defaults
direction="down"
count=10
delay=0.1
warpLabel="WarpTCP"
stdLabel="TCP"

# Get parameter values
required=0
while [[ $# -gt 0 ]]
do
   parm="$1"

   case "$parm" in
      --fileSize)
         let required=$required+1
         size="$2"
         shift
         ;;
      --warpHost)
         let required=$required+1
         host1="$2"
         shift
         ;;
      --warpPort)
         let required=$required+1
         port1="$2"
         shift
         ;;
      --standardHost)
         let required=$required+1
         host2="$2"
         shift
         ;;
      --standardPort)
         let required=$required+1
         port2="$2"
         shift
         ;;
      --ftpUser)
         let required=$required+1
         ftpUser="$2"
         shift
         ;;
      --ftpPassword)
         let required=$required+1
         ftpPass="$2"
         shift
         ;;
      --direction)
         case "$2" in
            up|down) direction="$2" ;;
            *) echo "Direction can only be 'up' or 'down'. Aborting"; exit 1 ;;
         esac
         shift
         ;;
      --count)
         count="$2"
         shift
         ;;
      --delay)
         delay="$2"
         shift
         ;;
      -h|--help)
         usage
         exit 0
         ;;
      *)
         echo "Unknown option: $1"
         echo "Aborting"
         echo "========"
         usage
         exit 1
      ;;
   esac
shift # Next argument or value
done

# Final parameter check
if [ $required -lt 7 ]; then
   echo "Missing parameters, aborting"
   usage;
   exit 1;
fi


# Output column headers
echo "Iteration,Direction,Server,Date + Time,TTFB,Total Time,Transfer Speed (bytes/second),Response Code,File Size (bytes),URL" 

# Begin iterating through FTP transfers
let i=$count-1
while [ $i -ge 0 ];
do
   # Generate timestamp for warp test
   ts=`date "+%Y-%m-%d %H:%M:%S"`

   # Test against warpHost
   if [ "x$direction" == "xup" ]; then
      res1=$i,upload,$warpLabel,`curl -w "$ts,%{time_starttransfer},%{time_total},%{speed_upload},%{http_code},%{size_upload},%{url_effective}\n" -o "/dev/null" -s -u ${ftpUser}:${ftpPass} -T ${size}_test.html ftp://${host1}:${port1}/pub/${size}-upload`
   else
      res1=$i,download,$warpLabel,`curl -w "$ts,%{time_starttransfer},%{time_total},%{speed_download},%{http_code},%{size_download},%{url_effective}\n" -o "/dev/null" -s -u ${ftpUser}:${ftpPass} ftp://${host1}:${port1}/${size}_test.html`
   fi

   # Output result and sleep
   echo "$res1"
   sleep $delay

   # Generate timestamp for standard test
   ts=`date "+%Y-%m-%d %H:%M:%S"`

   # Test against standard host
   if [ "x$direction" == "xup" ]; then
      res2=$i,upload,$stdLabel,`curl -w "$ts,%{time_starttransfer},%{time_total},%{speed_upload},%{http_code},%{size_upload},%{url_effective}\n" -o "/dev/null" -s -u ${ftpUser}:${ftpPass} -T ${size}_test.html ftp://${host2}:${port2}/pub/${size}-upload`
   else
      res2=$i,download,$stdLabel,`curl -w "$ts,%{time_starttransfer},%{time_total},%{speed_download},%{http_code},%{size_download},%{url_effective}\n" -o "/dev/null" -s -u ${ftpUser}:${ftpPass} ftp://${host2}:${port2}/${size}_test.html`
fi

   # Output result and sleep
   echo "$res2"
   sleep $delay
   let i=$i-1
done

exit 0
